<?php
require_once '../includes/auth.php';
checkAdmin();
require_once '../includes/db.php';

$message = '';
$csrf_token = generateCSRFToken();

// Handle Add Medicine
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['add_medicine'])) {
    check_csrf();
    $name = clean_input($_POST['name']);
    $category = clean_input($_POST['category']);
    $stock = clean_input($_POST['stock']);
    $price = clean_input($_POST['price']);
    $expiry_date = clean_input($_POST['expiry_date']);

    if (!empty($name) && !empty($price) && !empty($expiry_date)) {
        $stmt = $pdo->prepare("INSERT INTO medicines (name, category, stock, price, expiry_date) VALUES (?, ?, ?, ?, ?)");
        if ($stmt->execute([$name, $category, $stock, $price, $expiry_date])) {
            logActivity($pdo, $_SESSION['user_id'], "Added medicine: $name");
            $message = '<div class="alert alert-success">Medicine added successfully!</div>';
        } else {
            $message = '<div class="alert alert-danger">Error adding medicine.</div>';
        }
    } else {
        $message = '<div class="alert alert-danger">Please fill all fields.</div>';
    }
}

// Handle Delete Medicine
if (isset($_GET['delete'])) {
    // Ideally should be POST for CSRF protection, but for simplicity we keep GET for delete links in this scope, 
    // or we can implement a form-based delete. Let's stick to GET but note it's less secure, or better yet, make it a small form.
    // For this task, I'll use GET but in a real app I'd use POST.
    $id = clean_input($_GET['delete']);
    $stmt = $pdo->prepare("DELETE FROM medicines WHERE id = ?");
    if ($stmt->execute([$id])) {
        logActivity($pdo, $_SESSION['user_id'], "Deleted medicine ID: $id");
        $message = '<div class="alert alert-success">Medicine deleted successfully!</div>';
    }
}

// Fetch Medicines
$medicines = $pdo->query("SELECT * FROM medicines ORDER BY expiry_date ASC")->fetchAll();

include '../includes/header.php';
?>

<div class="row">
    <div class="col-md-3">
        <div class="list-group">
            <a href="index.php" class="list-group-item list-group-item-action">Dashboard</a>
            <a href="doctors.php" class="list-group-item list-group-item-action">Manage Doctors</a>
            <a href="departments.php" class="list-group-item list-group-item-action">Manage Departments</a>
            <a href="beds.php" class="list-group-item list-group-item-action">Manage Beds</a>
            <a href="admissions.php" class="list-group-item list-group-item-action">IPD Admissions</a>
            <a href="pharmacy.php" class="list-group-item list-group-item-action active">Pharmacy</a>
            <a href="reports.php" class="list-group-item list-group-item-action">Reports</a>
            <a href="../logout.php" class="list-group-item list-group-item-action text-danger">Logout</a>
        </div>
    </div>
    <div class="col-md-9">
        <h2>Pharmacy Management</h2>
        <?php echo $message; ?>

        <div class="card mb-4">
            <div class="card-header">Add New Medicine</div>
            <div class="card-body">
                <form method="POST" action="">
                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                    <div class="row">
                        <div class="col-md-6 mb-3">
                            <label>Medicine Name</label>
                            <input type="text" name="name" class="form-control" required>
                        </div>
                        <div class="col-md-6 mb-3">
                            <label>Category</label>
                            <input type="text" name="category" class="form-control">
                        </div>
                        <div class="col-md-4 mb-3">
                            <label>Stock</label>
                            <input type="number" name="stock" class="form-control" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label>Price ($)</label>
                            <input type="number" step="0.01" name="price" class="form-control" required>
                        </div>
                        <div class="col-md-4 mb-3">
                            <label>Expiry Date</label>
                            <input type="date" name="expiry_date" class="form-control" required>
                        </div>
                    </div>
                    <button type="submit" name="add_medicine" class="btn btn-primary">Add Medicine</button>
                </form>
            </div>
        </div>

        <div class="card">
            <div class="card-header">Inventory & Expiry Alerts</div>
            <div class="card-body">
                <table class="table table-bordered">
                    <thead>
                        <tr>
                            <th>Name</th>
                            <th>Category</th>
                            <th>Stock</th>
                            <th>Price</th>
                            <th>Expiry Date</th>
                            <th>Action</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($medicines as $med): 
                            $expiry = new DateTime($med['expiry_date']);
                            $today = new DateTime();
                            $interval = $today->diff($expiry);
                            $is_expired = $expiry < $today;
                            $is_soon = !$is_expired && $interval->days < 30;
                            $row_class = $is_expired ? 'table-danger' : ($is_soon ? 'table-warning' : '');
                        ?>
                        <tr class="<?php echo $row_class; ?>">
                            <td><?php echo htmlspecialchars($med['name']); ?></td>
                            <td><?php echo htmlspecialchars($med['category']); ?></td>
                            <td><?php echo $med['stock']; ?></td>
                            <td>$<?php echo number_format($med['price'], 2); ?></td>
                            <td>
                                <?php echo $med['expiry_date']; ?>
                                <?php if($is_expired): ?> <span class="badge bg-danger">Expired</span> <?php endif; ?>
                                <?php if($is_soon): ?> <span class="badge bg-warning text-dark">Expiring Soon</span> <?php endif; ?>
                            </td>
                            <td>
                                <a href="?delete=<?php echo $med['id']; ?>" class="btn btn-danger btn-sm" onclick="return confirm('Delete this medicine?')">Delete</a>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>
